<?php

use yii\helpers\Html;
use yii\helpers\Url;
use yii\web\View;


// Регистрируем CSS стили
$css = <<< CSS
/* Основные цвета */
:root {
    --primary-black: #121212;
    --secondary-black: #1a1a1a;
    --dark-gray: #2d2d2d;
    --light-gray: #444;
    --primary-pink: #e83e8c;
    --secondary-pink: #ff6b9d;
    --soft-pink: rgba(232, 62, 140, 0.1);
    --text-primary: #f8f9fa;
    --text-secondary: #b0b0b0;
    --border-color: #333;
    --hover-black: #252525;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
}

/* Контейнер админ-панели */
.admin-container {
    background-color: var(--primary-black);
    border-radius: 16px;
    padding: 30px;
    box-shadow: 0 15px 35px rgba(0, 0, 0, 0.5);
    border: 1px solid var(--border-color);
    margin: 0 auto;
    position: relative;
    overflow: hidden;
    max-width: 1200px;
    width: 100%;
}

.admin-container::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 4px;
    background: linear-gradient(90deg, var(--primary-pink), var(--secondary-pink));
    z-index: 10;
}

.admin-container::after {
    content: '';
    position: absolute;
    top: -50%;
    left: -50%;
    width: 200%;
    height: 200%;
    background: radial-gradient(circle, rgba(232, 62, 140, 0.05) 0%, rgba(232, 62, 140, 0) 70%);
    z-index: 0;
    pointer-events: none;
}

/* Заголовок админ-панели */
.admin-header {
    margin-bottom: 35px;
    position: relative;
    z-index: 1;
    text-align: center;
}

.admin-title {
    color: var(--text-primary);
    font-size: 36px;
    font-weight: 600;
    margin-bottom: 10px;
    background: linear-gradient(135deg, var(--primary-pink), var(--secondary-pink));
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
    background-clip: text;
    position: relative;
    display: inline-block;
}

.admin-title::after {
    content: '⚙️';
    position: absolute;
    right: -40px;
    top: 50%;
    transform: translateY(-50%);
    font-size: 24px;
}

.admin-subtitle {
    color: var(--text-secondary);
    font-size: 16px;
    margin-top: 10px;
}

/* Статистика */
.admin-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 20px;
    margin-bottom: 40px;
    position: relative;
    z-index: 1;
}

.stat-card {
    background-color: var(--secondary-black);
    border-radius: 12px;
    padding: 25px 20px;
    border: 1px solid var(--border-color);
    transition: all 0.3s ease;
    position: relative;
    overflow: hidden;
}

.stat-card:hover {
    transform: translateY(-5px);
    border-color: var(--primary-pink);
    box-shadow: 0 10px 25px rgba(232, 62, 140, 0.15);
}

.stat-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    width: 4px;
    height: 100%;
    background: linear-gradient(180deg, var(--primary-pink), var(--secondary-pink));
}

.stat-value {
    color: var(--text-primary);
    font-size: 32px;
    font-weight: 700;
    margin-bottom: 5px;
}

.stat-label {
    color: var(--text-secondary);
    font-size: 14px;
    text-transform: uppercase;
    letter-spacing: 1px;
}

/* Список заявок */
.requests-list {
    position: relative;
    z-index: 1;
}

.request-item {
    background-color: var(--secondary-black);
    border-radius: 12px;
    padding: 25px;
    margin-bottom: 20px;
    border: 1px solid var(--border-color);
    transition: all 0.3s ease;
    opacity: 0;
    animation: slideInRequest 0.5s ease-out forwards;
}

@keyframes slideInRequest {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.request-item:hover {
    border-color: var(--primary-pink);
    box-shadow: 0 8px 20px rgba(232, 62, 140, 0.1);
    transform: translateX(5px);
}

.request-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding-bottom: 15px;
    border-bottom: 1px solid var(--border-color);
}

.request-user {
    display: flex;
    align-items: center;
    gap: 15px;
}

.user-avatar {
    width: 50px;
    height: 50px;
    background: linear-gradient(135deg, var(--primary-pink), var(--secondary-pink));
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-weight: 600;
    font-size: 20px;
}

.user-info h4 {
    color: var(--text-primary);
    font-size: 18px;
    margin-bottom: 5px;
    font-weight: 600;
}

.user-info .user-phone {
    color: var(--text-secondary);
    font-size: 14px;
}

.request-game {
    text-align: right;
}

.game-name {
    color: var(--text-primary);
    font-size: 18px;
    font-weight: 600;
    margin-bottom: 5px;
}

.game-date {
    color: var(--text-secondary);
    font-size: 14px;
}

/* Статус заявки */
.request-status {
    display: inline-block;
    padding: 8px 16px;
    border-radius: 20px;
    font-size: 14px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 20px;
}

.status-pending {
    background-color: rgba(255, 193, 7, 0.1);
    color: var(--warning-color);
    border: 1px solid rgba(255, 193, 7, 0.3);
}

.status-approved {
    background-color: rgba(40, 167, 69, 0.1);
    color: var(--success-color);
    border: 1px solid rgba(40, 167, 69, 0.3);
}

.status-completed {
    background-color: rgba(23, 162, 184, 0.1);
    color: var(--info-color);
    border: 1px solid rgba(23, 162, 184, 0.3);
}

/* Детали заявки */
.request-details {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 15px;
    margin-bottom: 25px;
}

.detail-item {
    background-color: var(--dark-gray);
    padding: 12px 15px;
    border-radius: 8px;
    border-left: 3px solid var(--primary-pink);
}

.detail-label {
    color: var(--text-secondary);
    font-size: 12px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 5px;
}

.detail-value {
    color: var(--text-primary);
    font-size: 16px;
    font-weight: 500;
}

/* Кнопки действий */
.request-actions {
    display: flex;
    gap: 15px;
    justify-content: flex-end;
}

.btn-action {
    padding: 12px 25px;
    border-radius: 8px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    font-size: 14px;
    transition: all 0.3s ease;
    border: none;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    text-decoration: none;
    position: relative;
    overflow: hidden;
}

.btn-action::after {
    content: '';
    position: absolute;
    top: 0;
    left: -100%;
    width: 100%;
    height: 100%;
    background: linear-gradient(90deg, transparent, rgba(255, 255, 255, 0.2), transparent);
    transition: left 0.7s ease;
}

.btn-action:hover::after {
    left: 100%;
}

.btn-approve {
    background: linear-gradient(135deg, var(--success-color), #20c997);
    color: white;
    box-shadow: 0 4px 15px rgba(40, 167, 69, 0.3);
}

.btn-approve:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(40, 167, 69, 0.4);
}

.btn-complete {
    background: linear-gradient(135deg, var(--info-color), #0dcaf0);
    color: white;
    box-shadow: 0 4px 15px rgba(23, 162, 184, 0.3);
}

.btn-complete:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(23, 162, 184, 0.4);
}

.btn-decline {
    background: linear-gradient(135deg, var(--danger-color), #e35d6a);
    color: white;
    box-shadow: 0 4px 15px rgba(220, 53, 69, 0.3);
}

.btn-decline:hover {
    transform: translateY(-3px);
    box-shadow: 0 8px 20px rgba(220, 53, 69, 0.4);
}

/* Иконки на кнопках */
.btn-action::before {
    font-size: 16px;
}

.btn-approve::before {
    content: '✓';
}

.btn-complete::before {
    content: '✓';
}

.btn-decline::before {
    content: '✕';
}

/* Анимация загрузки */
@keyframes pulse {
    0% { opacity: 1; }
    50% { opacity: 0.5; }
    100% { opacity: 1; }
}

.loading-pulse {
    animation: pulse 1.5s infinite;
}

/* Пустой список */
.empty-list {
    text-align: center;
    padding: 60px 20px;
    background-color: var(--secondary-black);
    border-radius: 12px;
    border: 2px dashed var(--border-color);
}

.empty-icon {
    font-size: 48px;
    margin-bottom: 20px;
    color: var(--primary-pink);
}

.empty-text {
    color: var(--text-secondary);
    font-size: 18px;
}

/* Адаптивность */
@media (max-width: 768px) {
    .admin-container {
        padding: 20px;
    }
    
    .admin-title {
        font-size: 28px;
    }
    
    .admin-stats {
        grid-template-columns: 1fr;
    }
    
    .request-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .request-game {
        text-align: left;
        width: 100%;
    }
    
    .request-actions {
        flex-wrap: wrap;
        justify-content: center;
    }
    
    .btn-action {
        flex: 1;
        min-width: 120px;
        justify-content: center;
    }
    
    .request-details {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 480px) {
    .admin-container {
        padding: 15px;
    }
    
    .request-item {
        padding: 20px;
    }
    
    .user-avatar {
        width: 40px;
        height: 40px;
        font-size: 16px;
    }
    
    .user-info h4 {
        font-size: 16px;
    }
    
    .game-name {
        font-size: 16px;
    }
}
CSS;

$this->registerCss($css);

// Регистрируем JavaScript
$js = <<< JS
$(document).ready(function() {
    // Анимация появления элементов
    $('.request-item').each(function(index) {
        $(this).css('animation-delay', (index * 0.1) + 's');
    });
    
    // Подсчет статистики
    function updateStats() {
        var totalRequests = $('.request-item').length;
        var pendingRequests = $('.status-pending').length;
        var approvedRequests = $('.status-approved').length;
        var completedRequests = $('.status-completed').length;
        
        $('.stat-total').text(totalRequests);
        $('.stat-pending').text(pendingRequests);
        $('.stat-approved').text(approvedRequests);
        $('.stat-completed').text(completedRequests);
    }
    
    updateStats();
    

    
    function updateRequestStatus(button) {
        var requestItem = button.closest('.request-item');
        var statusElement = requestItem.find('.request-status');
        
        if (button.hasClass('btn-approve')) {
            statusElement
                .removeClass('status-pending')
                .addClass('status-approved')
                .text('Заявка одобрена');
        } else if (button.hasClass('btn-complete')) {
            statusElement
                .removeClass('status-approved status-pending')
                .addClass('status-completed')
                .text('Заказ завершен');
        }
        
        // Обновляем статистику
        updateStats();
        
        // Показываем уведомление
        showNotification('Статус обновлен успешно!');
    }
    
    function showNotification(message) {
        var notification = $('<div class="notification">')
            .text(message)
            .css({
                'position': 'fixed',
                'top': '20px',
                'right': '20px',
                'background': 'linear-gradient(135deg, var(--primary-pink), var(--secondary-pink))',
                'color': 'white',
                'padding': '15px 25px',
                'borderRadius': '8px',
                'boxShadow': '0 5px 15px rgba(0,0,0,0.3)',
                'zIndex': '9999',
                'transform': 'translateX(150%)',
                'transition': 'transform 0.3s ease'
            })
            .appendTo('body');
        
        notification.css('transform', 'translateX(0)');
        
        setTimeout(function() {
            notification.css('transform', 'translateX(150%)');
            setTimeout(function() {
                notification.remove();
            }, 300);
        }, 3000);
    }
    
    // Фильтрация заявок
    $('.filter-btn').on('click', function() {
        var filter = $(this).data('filter');
        
        $('.filter-btn').removeClass('active');
        $(this).addClass('active');
        
        if (filter === 'all') {
            $('.request-item').show();
        } else {
            $('.request-item').hide();
            $('.status-' + filter).closest('.request-item').show();
        }
    });
    
    // Автоматическое обновление каждые 30 секунд
    setInterval(function() {
        // В реальном приложении здесь будет AJAX запрос для обновления данных
        console.log('Обновление данных...');
    }, 30000);
});
JS;

$this->registerJs($js, \yii\web\View::POS_READY);


// Подсчет статистики
$totalRequests = count($requests);
$pendingRequests = array_filter($requests, fn($r) => $r->status === 'Ожидает рассмотрения');
$approvedRequests = array_filter($requests, fn($r) => $r->status === 'Заявка одобрена');
$completedRequests = array_filter($requests, fn($r) => $r->status === 'Заказ завершен');
?>

<div class="admin-container">
    <div class="admin-header">
        <h1 class="admin-title">Админ-панель</h1>
        <p class="admin-subtitle">Управление заявками на участие в играх</p>
    </div>

    <!-- Статистика -->
    <div class="admin-stats">
        <div class="stat-card">
            <div class="stat-value stat-total"><?= $totalRequests ?></div>
            <div class="stat-label">Всего заявок</div>
        </div>
        <div class="stat-card">
            <div class="stat-value stat-pending"><?= count($pendingRequests) ?></div>
            <div class="stat-label">Ожидают рассмотрения</div>
        </div>
        <div class="stat-card">
            <div class="stat-value stat-approved"><?= count($approvedRequests) ?></div>
            <div class="stat-label">Одобрено</div>
        </div>
        <div class="stat-card">
            <div class="stat-value stat-completed"><?= count($completedRequests) ?></div>
            <div class="stat-label">Завершено</div>
        </div>
    </div>

    <!-- Список заявок -->
    <div class="requests-list">
        <?php if (!empty($requests)): ?>
            <?php foreach ($requests as $r): ?>
                <?php
                // Определяем класс статуса
                $statusClass = 'status-pending';
                if ($r->status === 'Заявка одобрена') {
                    $statusClass = 'status-approved';
                } elseif ($r->status === 'Заказ завершен') {
                    $statusClass = 'status-completed';
                }

                // Получаем первую букву имени для аватарки
                $avatarLetter = mb_substr($r->user->fio, 0, 1, 'UTF-8');
                ?>

                <div class="request-item">
                    <div class="request-header">
                        <div class="request-user">
                            <div class="user-avatar"><?= $avatarLetter ?></div>
                            <div class="user-info">
                                <h4><?= Html::encode($r->user->fio) ?></h4>
                                <div class="user-phone"><?= Html::encode($r->user->phone) ?></div>
                            </div>
                        </div>
                        <div class="request-game">
                            <div class="game-name"><?= Html::encode($r->game_name) ?></div>
                            <div class="game-date"><?= date('d.m.Y', strtotime($r->created_at)) ?></div>
                        </div>
                    </div>

                    <div class="request-status <?= $statusClass ?>">
                        <?= Html::encode($r->status) ?>
                    </div>

                    <div class="request-details">
                        <div class="detail-item">
                            <div class="detail-label">ID заявки</div>
                            <div class="detail-value">#<?= $r->id ?></div>
                        </div>
                        <div class="detail-item">
                            <div class="detail-label">Дата создания</div>
                            <div class="detail-value"><?= Yii::$app->formatter->asDatetime($r->created_at) ?></div>
                        </div>
                        <div class="detail-item">
                            <div class="detail-label">Вид игры</div>
                            <div class="detail-value"><?= Html::encode($r->game_type) ?></div>
                        </div>
                        <div class="detail-item">
                            <div class="detail-label">Время игры</div>
                            <div class="detail-value"><?= Html::encode($r->game_time) ?></div>
                        </div>
                    </div>

                    <div class="request-actions">
                        <?php if ($r->status === 'Новая'): ?>
                            <a href="<?= Url::to(['admin/update-status', 'id'=>$r->id, 'status'=>'Заявка одобрена']) ?>"
                               class="btn btn-success btn-sm">
                                Одобрить
                            </a>

                        <?php elseif ($r->status === 'Заявка одобрена'): ?>
                            <a href="<?= Url::to(['admin/update-status', 'id'=>$r->id, 'status'=>'Заказ завершен']) ?>"
                               class="btn btn-warning btn-sm">
                                Завершить
                            </a>

                        <?php else: ?>
                            <span class="text-muted">—</span>
                        <?php endif; ?>
                    </div>
                </div>
            <?php endforeach; ?>
        <?php else: ?>
            <div class="empty-list">
                <div class="empty-icon">📭</div>
                <div class="empty-text">Нет заявок для отображения</div>
            </div>
        <?php endif; ?>
    </div>
</div>